<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Exception;
use SebastianBergmann\Type\TrueType;
use stdClass;

class UserController extends Controller
{
    
    public function admin_authenticate(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required:email',
                'password' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $credentials = $request->only('email', 'password');
            
            if (!auth()->attempt($credentials)) {
                $response = ['status' => false, 'message' => "Invalid Credentials"];
                return response($response, 400);
            } else {
                $user_id = auth()->user()->id;
                $data = User::where('id', $user_id)->first();

                if ($data->role != 'ADMIN') {
                    $response = ['status' => false, 'data' => $data, 'message' => "Invalid credentials!"];
                    return response($response, 400);
                } else {
                    $data->device_token = $request->device_token?$request->device_token:"";
                    $data->save();
                    $data->token = auth()->user()->createToken('API Token')->accessToken;
                    unset($data->profile_picture);
                    $response = ['status' => true, 'data' => $data, 'message' => "Account login successfully!"];
                    return response($response, 200);
                }
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function dashboard(Request $request)
    {
        try {
            $data = array();
            $employees = DB::table('user')->join('employee', 'user.id', '=', 'employee.user_id')
                ->where('user.role','=','USER')
                ->orderBy('user.id', 'desc')
                ->limit(5)
                ->get();
            $data['recent_employees'] = $employees;
            $response = ['status' => true, 'data' => $data, 'message' => "Success"];
            return response($response, 200);
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_employees(Request $request)
    {
        try {
            $data = array();
            $employees = DB::table('user')->join('employee', 'user.id', '=', 'employee.user_id')
                ->where('user.role','=','USER')
                ->orderBy('user.id', 'desc')
                ->get();
            $data['employees'] = $employees;
            $response = ['status' => true, 'data' => $data, 'message' => "Success"];
            return response($response, 200);
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }
    //
    public function register(Request $request)
    {
        try {
            $validator = Validator::make(
                $request->all(),
                [
                    'name' => 'required',
                    'email' => 'required|unique:user',
                    'password' => 'required|min:6',
                ]
            );
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $code = mt_rand(000000, 999999);
            $user = new User();
            $user->name = $request->name;
            $user->email = $request->email;
            $user->role = 'USER';
            $user->password = Hash::make($request->password);
            $user->code = $code;
            $user->password_code = '';
            $user->device_token = $request->device_token?$request->device_token:'';

            $user->save();
            $data = User::where('id', $user->id)->first();
            
            if ($user == true) {
                $response = ['status' => true, 'data' => $data, 'message' => "Account created successfully. Thank you!"];
                return response($response, 200);
            }else {
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function social_register(Request $request)
    {
        try {
            $validator = Validator::make(
                $request->all(),
                [
                    'social_id' => 'required',
                    'platform' => 'required',
                    'name' => 'required',
                    //'email' => 'unique:users',
                    //'phone' => 'unique:users',
                ]
            );
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $code = mt_rand(000000, 999999);
            $user_info = User::where('social_id', $request->social_id)->first();
            if($user_info){
                $data = $user_info;
                //user already exist
                if(!auth()->loginUsingId($data->id)){    
                    $response = ['status' => false, 'message' => "Something went wrong while accessing the account."];
                    return response($response, 400);
                } else {
                    $data['token'] = auth()->user()->createToken('API Token')->accessToken;
                    $data['new_user'] = false;
                    $response = ['status' => true, 'data' => $data, 'message' => "Account login successfully!"];
                    return response($response, 200);
                }
            }else{
                //create new user
                $user = new User();
                $user->name = $request->name;
                $user->email = $request->email?strtolower($request->email):"";
                $user->phone = $request->phone?$request->phone:"";
                $user->role = 'User';
                $user->password = Hash::make('password');
                $user->code = $code;
                $user->social_id = $request->social_id;
                $user->is_verified = 1;
                $user->platform = $request->platform;
                $user->device_token = $request->device_token?$request->device_token:'';
                $user->save();
                $data = User::where('social_id', $request->social_id)->first();
                if ($user == true) {
                    if(!auth()->loginUsingId($data->id)){
                        $response = ['status' => false, 'message' => "Email or password is invalid. Thank you!"];
                        return response($response, 400);
                    } else {
                        $data['token'] = auth()->user()->createToken('API Token')->accessToken;
                        $data['new_user'] = true;
                        $response = ['status' => true, 'data' => $data, 'message' => "Account login successfully!"];
                        return response($response, 200);
                    }
                }
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function authenticate(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required:email',
                'password' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $credentials = $request->only('email', 'password');
            
            if (!auth()->attempt($credentials)) {
                $response = ['status' => false, 'message' => "Invalid Credentials. Thank you!"];
                return response($response, 400);
            } else {
                $user_id = auth()->user()->id;
                $data = User::where('id', $user_id)->first();

                if ($data->status == 0) {
                    $response = ['status' => false, 'data' => $data, 'message' => "Your account is not verified. Please verify your account. Thank you!"];
                    return response($response, 400);
                } else {
                    $data->device_token = $request->device_token?$request->device_token:"";
                    $data->save();
                    $data->token = auth()->user()->createToken('API Token')->accessToken;
                    $response = ['status' => true, 'data' => $data, 'message' => "Account login successfully!"];
                    return response($response, 200);
                }
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function forgot_password(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }

            $email = $request->email;
            $found = User::where('email', $email)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $code = mt_rand(000000, 999999);
            // $mail = Mail::send('emails.forgetEmail', ['code' => $code], function ($message) use ($request) {
            //     $message->to($request->email);
            //     $message->subject('Forget Password');
            // });
            // if ($mail == true) {
            $user = User::where('email', $email)->first();
            $user->password_code = $code;
            $user->save();
            if ($user == true) {
                $response = ['status' => true, 'data' => $user, 'message' => "An OTP with a reset code sent successfully!"];
                return response($response, 200);
            }else {
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function verification(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required',
                'code' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user = User::where('id', $request->id)->first();
            if($user){
                if ($user->is_verified == 1) {
                    $response = ['status' => false, 'data' => null, 'message' => "Already verified. Thank you!"];
                    return response($response, 200);
                }else{
                    if ($user->code == $request->code) {
                        if(!auth()->loginUsingId($request->id)){ 
                            $response = ['status' => false, 'message' => "Invalid account!"];
                            return response($response, 400);
                        } else {
                            if ($user->role == "USER") {
                                $data = User::where('id', $request->id)->first();
                            }
                            // if ($user->role == "Salon") {
                            //     $data = User::where('id', $request->id)->first(['id', 'name', 'email', 'phone', 'profile_image', 'code', 'role', 'account_status', 'email_status', 'salon_name_en', 'salon_name_ar', 'commercial_registration_number', 'certificate', 'category', 'iban', 'country', 'city', 'average_orders', 'service_type', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday', 'shift', 'latitude', 'longitude', 'created_at', 'updated_at']);
                            // }
                            // if ($user->role == "Staff") {
                            //     $data = User::where('id', $request->id)->first(['id', 'name', 'email', 'phone', 'profile_image', 'code', 'role', 'account_status', 'email_status', 'salon_name_en', 'salon_name_ar', 'commercial_registration_number', 'certificate', 'category', 'iban', 'country', 'city', 'average_orders', 'service_type', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday', 'shift', 'latitude', 'longitude', 'created_at', 'updated_at']);
                            // }
                            $data->is_verified = '1';
                            $data->save();
                            $data['token'] =  Auth::user()->createToken('API Token')->accessToken;
                            $response = ['status' => true, 'data' => $data, 'message' => "Account verified successfully!"];
                            return response($response, 200);
                        }
                    }else{
                        $response = ['status' => false, 'data' => null, 'message' => "Code is Invalid!"];
                        return response($response, 400);
                    }
                }
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Not found."];
                return response($response, 200);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function update_password(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required',
                'code' => 'required',
                'password' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = $request->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $user = User::where('id', $id)->first();
            if($user->password_code == $request->code){
                $user->password = Hash::make($request->password);
                $user->save();
                if($user == true){
                    $response = ['status' => true, 'data' => $user, 'message' => "Password updated successfully!"];
                    return response($response, 200);
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                    return response($response, 400);
                }
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Invalid code"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'gender' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->gender = $request->gender;
                $profile->save();    
            }else{
                //new profile
                $profile = new UserProfile();
                $profile->user_id = $id;
                $profile->gender = $request->gender;
                $profile->save();
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_guardian(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'fullname' => 'required',
                'email' => 'required',
                'phone' => 'required',
                'relation' => 'required',
                'language' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile

                //guardian as user
                $check_guardian = User::where('email', $request->email)->where('role','GUARDIAN')->first();
                if(!$check_guardian){
                    $guardian_user = new User();
                    $guardian_user->name = $request->fullname;
                    $guardian_user->email = $request->email;
                    $guardian_user->phone = $request->phone;
                    $guardian_user->role = 'GUARDIAN';
                    $guardian_user->is_verified = 1;
                    $guardian_user->password = Hash::make("12345678");
                    $guardian_user->save();
                }else{
                    if($profile->guardian_user_id != $check_guardian->id){
                        $response = ['status' => false, 'data' => null, 'message' => "This guardian is already associated!"];
                        return response($response, 400);
                    }
                }
                //guardian profile
                $profile->guardian_full_name = $request->fullname;
                $profile->guardian_email = $request->email;
                $profile->guardian_phone = $request->phone;
                $profile->guardian_relation = $request->relation;
                $profile->guardian_language = $request->language;
                $profile->guardian_national_id = $request->national_id;
                if(isset($guardian_user)){
                    $profile->guardian_user_id = $guardian_user->id;
                }
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_about(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'age' => 'required',
                'dob' => 'required',
                'marital_status' => 'required',
                'ethnicity' => 'required',
                'location' => 'required',
                'profession' => 'required',
                'last_qualification' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            $user = USer::where('id', $id)->first();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->age = $request->age;
                $profile->dob = $request->dob;
                $profile->marital_status = $request->marital_status;
                $profile->ethnicity = $request->ethnicity;
                $profile->location = $request->location;
                $profile->profession = $request->profession;
                $profile->last_qualification = $request->last_qualification;
                $profile->children = $request->children?$request->children:"0";
                $profile->save();
                $has_images = false;
                if ($request->hasfile('profile_picture')) {
                    //$imageName = env('APP_URL') . 'uploads/profile/' . time() . '.' . $request->profile_picture->extension();
                    $imageNameProfile = uniqid() . '.' . $request->profile_picture->extension();
                    //$category->profile_picture = $imageName;
                    $request->profile_picture->move(public_path('uploads/profile'),  $imageNameProfile);
                    $user->profile_picture = $imageNameProfile;
                    $has_images = true;
                }   
                if ($request->hasfile('cover_picture')) {
                    //$imageName = env('APP_URL') . 'uploads/profile/' . time() . '.' . $request->profile_picture->extension();
                    $imageNameCover = uniqid() . '.' . $request->cover_picture->extension();
                    //$category->profile_picture = $imageName;
                    $request->cover_picture->move(public_path('uploads/cover'),  $imageNameCover);
                    $user->cover_picture = $imageNameCover;
                    $has_images = true;
                }    
                if($has_images == true){
                    $user->save();
                }
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_height(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'height' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->height = $request->height;
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_appearance(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'appearance' => 'required',
                'body' => 'required',
                'nature' => 'required',
                'weight' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->appearance = $request->appearance;
                $profile->body = $request->body;
                $profile->nature = $request->nature;
                $profile->weight = $request->weight;
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_opento(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'open_to' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->open_to = $request->open_to;
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_islam(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'practicing_since' => 'required',
                'islam' => 'required',
                'prayers' => 'required',
                'revert' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->practicing_since = $request->practicing_since;
                $profile->islam = $request->islam;
                $profile->prayers = $request->prayers;
                $profile->revert = $request->revert;
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_self(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'desc_about_me' => 'required',
                'desc_living' => 'required',
                'desc_daily_attire' => 'required',
                'desc_hobbies_interest' => 'required',
                'desc_future_spouse' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->desc_about_me = $request->desc_about_me;
                $profile->desc_living = $request->desc_living;
                $profile->desc_daily_attire = $request->desc_daily_attire;
                $profile->desc_hobbies_interest = $request->desc_hobbies_interest;
                $profile->desc_future_spouse = $request->desc_future_spouse;
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_religion(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'desc_studied_islam' => 'required',
                'desc_books_studied' => 'required',
                'desc_pray_masjid' => 'required',
                'desc_scholars' => 'required',
                'desc_hijraah_plan' => 'required',
                'desc_successful_muslim' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $id = Auth::user()->id;
            $found = User::where('id', $id)->count();
            if ($found == 0) {
                $response = ['status' => false, 'data' => null, 'message' => "Account not found!"];
                return response($response, 400);
            }
            $profile = UserProfile::where("user_id", $id)->first();
            if($profile){
                //existing profile
                $profile->desc_studied_islam = $request->desc_studied_islam;
                $profile->desc_books_studied = $request->desc_books_studied;
                $profile->desc_pray_masjid = $request->desc_pray_masjid;
                $profile->desc_scholars = $request->desc_scholars;
                $profile->desc_hijraah_plan = $request->desc_hijraah_plan;
                $profile->desc_successful_muslim = $request->desc_successful_muslim;
                $profile->save();    
            }else{
                //new profile
                $response = ['status' => false, 'data' => null, 'message' => "Please start from step one"];
                return response($response, 400);
            }
            if($profile == true){
                $response = ['status' => true, 'data' => $profile, 'message' => "Success"];
                return response($response, 200);
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong. Please try again later. Thank you!"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_interest(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                //'images' => 'required'
                'title' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $data = array();
            $i = 0;
            foreach($request->title as $title){
                $image_name = '';
                if($request->hasfile('images')){
                    if($request->images[$i]){
                        $imagefile = $request->images[$i];
                        $image_name = uniqid() . '.' . $imagefile->extension();
                        $imagefile->move(public_path('uploads/interest'),  $image_name);
                    }
                }
                $interest = new UserInterest();
                $interest->user_id = $user_id;
                $interest->title = $title;
                $interest->picture = $image_name;
                $interest->save();
                $data[] = $interest;
                $i++;
            } 
            // if ($request->hasfile('images')) {
            //     $interest_images = [];
            //     $i = 0;
            //     foreach($request->images as $imagefile) {
            //         $image_name = uniqid() . '.' . $imagefile->extension();
            //         $imagefile->move(public_path('uploads/interest'),  $image_name);
            //         $interest_images[$i]['image'] = $image_name;
            //         //$interest_images[$i]['title'] = isset($request->title[$i])?$request->title[$i]:'';
            //         $i++;
            //     };
            // }
            
            // foreach($interest_images as $interest_image){
            //     $interest = new UserInterest();
            //     $interest->user_id = $user_id;
            //     $interest->title = $interest_image['title'];
            //     $interest->picture = $interest_image['image'];
            //     $interest->save();
            //     $data[] = $interest;
            // }
            $response = ['status' => true, 'data' => $data, 'message' => 'Success'];
            return response($response, 400);
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function profile_detail($id){
        try {
            $user_id = Auth::user()->id;
            $user_detail = $this->get_user_profile($id);
            $personality_test = new stdClass();
            $personality_test->type = "ESJF-14";
            $personality_test->class = "Potential";
            $personality_test->brief = "Very energetic and confident look";
            $personality_test->energetic = 60;
            $personality_test->leader = 60;
            $personality_test->decision = 60;
            $personality_test->committed = 60;
            $personality_test->description_heading = "ESJF - Ambassador";
            $personality_test->description_detail = "ESFJS are caring, loyal, and attentive to others' needs,
            ensuring the people around them are taken care of.
            Often the social butterflies and center of a community,
            they empathize easily with others. People are drawn to
            their social grace, warmth, and no-nonsense practical
            nature. They do not shy away from commitments and
            responsibility and find possible solutions to help those
            struggling.
            For ESFJs, the best way to learn is through other people.
            Meeting them, hearing their stories, and knowing about
            their beliefs can teach people everything.";
            $personality_test->strength = array('strong practical skills','sensitive','loyal');
            $personality_test->weakness = array('often too needy','inflexible','too selfless');
            $personality_test->attracted_by = array('supportive','realistic','caring','dependable');
            $user_detail->personality = $personality_test;
            $response = ['status' => true, 'data' => $user_detail, 'message' => "Success"];
            return response($response, 200);
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_user_profile($id){
        $user_profile = User::where('id', $id)->with('profile')->with('interest')->first();
        return $user_profile;
    }

    public function get_user_profile_short($logged_in_id, $offset = 0, $limit = 20, $gender, $filter){
        //dd($filter);
        //->leftJoin('boost_profile', 'user.id','=','boost_profile.user_id')
        $user_profile = DB::table('user')->join('user_profile', 'user.id', '=', 'user_profile.user_id')
        ->leftJoin('boost_profile', function ($join) {
            $join->on('user.id', '=', 'boost_profile.user_id')
            ->where(DB::raw("DATE(boost_profile.valid_from) "),"<=",DB::raw("DATE(now())"))
            ->where(DB::raw("DATE(boost_profile.valid_to) "),">=",DB::raw("DATE(now())"));
        })
        ->where('user.id','!=',$logged_in_id)
        ->where('user_profile.gender', '!=', $gender)
        ->select('boost_profile.id as boost_id','user.*','user_profile.age','user_profile.height','user_profile.location','user_profile.profession','user_profile.islam','user_profile.desc_about_me','user_profile.ethnicity','user_profile.marital_status','user_profile.gender')
        ->orderBy('boost_profile.id', 'desc')
        ->offset($offset)
        ->limit($limit);
        if(isset($filter['age_from'],$filter['age_to']) && $filter['age_from'] > 0 && $filter['age_to'] > 0){
            $user_profile = $user_profile->where('user_profile.age','>=',(int)$filter['age_from']);
            $user_profile = $user_profile->where('user_profile.age','<=',(int)$filter['age_to']);
        }
        if(isset($filter['height_from'],$filter['height_to']) && $filter['height_from'] > 0 && $filter['height_to'] > 0){
            $user_profile = $user_profile->where('user_profile.height','>=',(int)$filter['height_from']);
            $user_profile = $user_profile->where('user_profile.height','<=',(int)$filter['height_to']);
        }
        if(isset($filter['marital_status']) && $filter['marital_status'] != ''){
            $user_profile = $user_profile->where('user_profile.marital_status','=',$filter['marital_status']);
        }
        if(isset($filter['caste']) && $filter['caste'] != ''){
            $user_profile = $user_profile->where('user_profile.islam','=',$filter['caste']);
        }
        $user_profile = $user_profile->get();
        // print_r($user_profile);
        // exit;
        return $user_profile;
    }

    public function send_request(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'addressee_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $user_info = User::where('id', $user_id)->first();
            if($user_info->request_remaining > 0){
                $request_exist = UserRequest::where('addressee_id', $request->addressee_id)->where('requester_id', $user_id)->count();
                if($request_exist){
                    //request already exist
                    $response = ['status' => false, 'data' => null, 'message' => "Request already sent earlier!"];
                    return response($response, 400);
                }else{
                    //new request
                    $user_request = new UserRequest();
                    $user_request->requester_id = $user_id;
                    $user_request->addressee_id = $request->addressee_id;
                    $user_request->status_code = 'R';
                    $user_request->specifier_id = $user_id;
                    $user_request->save();
                    if($user_request == true){
                        //SUBTRACT REQUEST
                        $user_info->request_remaining = $user_info->request_remaining - 1;
                        $user_info->save();
                        $response = ['status' => true, 'data' => $user_request, 'message' => "Request sent successfully"];
                        return response($response, 200);
                    }else{
                        $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later!"];
                        return response($response, 400);
                    }
                }
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Request cannot be sent. You do not have enough requests remaining."];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function accept_request(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'request_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $request_db = UserRequest::where('id', $request->request_id)->first();
            if($request_db){
                //request exist
                if($request_db->addressee_id == $user_id){
                    $accept_request = new UserRequest();
                    $accept_request->requester_id = $request_db->requester_id;
                    $accept_request->addressee_id = $request_db->addressee_id;
                    $accept_request->specifier_id = $user_id;
                    $accept_request->status_code = 'A';
                    $accept_request->save();
                    if($accept_request == true){
                        $response = ['status' => true, 'data' => $accept_request, 'message' => "Request Accepted!"];
                        return response($response, 400);    
                    }else{
                        $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                        return response($response, 400);
                    }
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                    return response($response, 400);
                }
            }else{
                //request not exist
                $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function decline_request(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'request_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $request_db = UserRequest::where('id', $request->request_id)->first();
            if($request_db){
                //request exist
                if($request_db->addressee_id == $user_id){
                    $accept_request = new UserRequest();
                    $accept_request->requester_id = $request_db->requester_id;
                    $accept_request->addressee_id = $request_db->addressee_id;
                    $accept_request->specifier_id = $user_id;
                    $accept_request->status_code = 'D';
                    $accept_request->save();
                    if($accept_request == true){
                        $response = ['status' => true, 'data' => $accept_request, 'message' => "Request Declined!"];
                        return response($response, 200);    
                    }else{
                        $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                        return response($response, 400);
                    }
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                    return response($response, 400);
                }
            }else{
                //request not exist
                $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_request(){
        DB::enableQueryLog();
        $user_id = Auth::user()->id;

        //REQUESTS SENT BY THE USER
        $request_sent = DB::select("SELECT r1.*, u.name as addressee_name, u.email as addressee_email, u.phone as addressee_phone, u.profile_picture as addressee_profile_picture
        FROM request r1
        INNER JOIN 
        (
            SELECT id, MAX(created_at) MaxCreatedAt, addressee_id
            FROM request
            GROUP BY addressee_id
        ) r2
        ON r1.addressee_id = r2.addressee_id
        AND r1.created_at = r2.MaxCreatedAt
        LEFT JOIN `user` u ON u.id = r1.addressee_id
        WHERE requester_id = ? AND status_code = 'R'
        ORDER BY r1.created_at DESC",[$user_id]);

        //REQUESTS RECEIVED BY THE USER
        $request_received = DB::select("SELECT r1.*, u.name as requester_name, u.email as requester_email, u.phone as requester_phone, u.profile_picture as requester_profile_picture
        FROM request r1
        INNER JOIN 
        (
            SELECT id, MAX(created_at) MaxCreatedAt, addressee_id
            FROM request
            GROUP BY requester_id
        ) r2
        ON r1.addressee_id = r2.addressee_id
        AND r1.created_at = r2.MaxCreatedAt
        LEFT JOIN `user` u ON u.id = r1.requester_id
        WHERE r1.addressee_id = ? AND status_code = 'R'
        ORDER BY r1.created_at DESC",[$user_id]);

        $data = array(
            "request_sent" => $request_sent,
            "request_received" => $request_received,
        );
        $response = ['status' => true, 'data' => $data, 'message' => "Success"];
        return response($response, 200);
    }

    public function get_all_question(){
        try {
            $questions = PersonalityQuestion::get();
            //dd($questions);
            $response = ['status' => true, 'data' => $questions, 'message' => "Success"];
            return response($response, 200);
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function personality_test(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'question' => 'required',
                'question_id' => 'required',
                'answer' => 'required',
                'type' => 'required',
                'point' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $personality_test = new PersonalityTest();
            $personality_test->user_id = $user_id;
            $personality_test->question = $request->question;
            $personality_test->question_id = $request->question_id;
            $personality_test->answer = $request->answer;
            $personality_test->type = $request->type;
            $personality_test->point = $request->point;
            $personality_test->status = 1;
            $personality_test->save();
            if($personality_test == true){
                if(isset($request->is_completed)){
                    //calculate personality
                    $personality_type = '';
                    $personality_tests = PersonalityTest::where('user_id', $user_id)->get();
                    $E = 0; $I = 0; $S = 0; $N = 0; $T = 0; $F = 0; $J = 0; $P = 0;
                    foreach($personality_tests as $pt){
                        if($pt->type == 'E') $E=$E+$pt->point;
                        if($pt->type == 'I') $I=$I+$pt->point;
                        if($pt->type == 'S') $S=$S+$pt->point;
                        if($pt->type == 'N') $N=$N+$pt->point;
                        if($pt->type == 'T') $T=$T+$pt->point;
                        if($pt->type == 'F') $F=$F+$pt->point;
                        if($pt->type == 'J') $J=$J+$pt->point;
                        if($pt->type == 'P') $P=$P+$pt->point;
                    }
                    //E & I
                    if($E > $I) $personality_type .= 'E';
                    else if($E < $I) $personality_type .= 'I';
                    else $personality_type .= 'I';
                    //S & N
                    if($S > $N) $personality_type .= 'S';
                    else if($S < $N) $personality_type .= 'N';
                    else $personality_type .= 'N';
                    //T & F
                    if($T > $F) $personality_type .= 'T';
                    else if($T < $F) $personality_type .= 'F';
                    else $personality_type .= 'T';
                    //J & P
                    if($J > $P) $personality_type .= 'J';
                    else if($J < $P) $personality_type .= 'P';
                    else $personality_type .= 'P';
                    $user_profile = UserProfile::where('user_id', $user_id)->first();
                    if($user_profile){
                        $user_profile->personality_type = $personality_type;
                        $user_profile->save();
                    }
                }
                $response = ['status' => true, 'data' => $personality_test, 'message' => "Success!"];
                return response($response, 400);    
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
            } 
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_packages(Request $request){
        try {
            $packages = Package::orderBy('id', 'asc')->get();
            $response = ['status' => true, 'data' => $packages, 'message' => "Success!"];
            return response($response, 400);    
             
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function create_post(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required',
                'content' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $post = new Post();
            $post->user_id = $user_id;
            $post->title = $request->title;
            $post->content = $request->content;
            $post->status = 1;
            $post->save();
            if($post == true){
                $images = array();
                if($files = $request->file('images')){
                    foreach($files as $file){
                        $image_name = uniqid() . '.' . $file->extension();
                        $file->move(public_path('uploads/post'),$image_name);
                        $images[] = $image_name;
                        $post_image = new PostImages();
                        $post_image->post_id = $post->id;
                        $post_image->image_path = $image_name;
                        $post_image->status = 1;
                        $post_image->save();
                    }
                }
                $response = ['status' => true, 'data' => $post, 'message' => "Success!"];
                return response($response, 400);    
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
            } 
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function edit_post(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required',
                'title' => 'required',
                'content' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $post_id = $request->id;
            $user_id = Auth::user()->id;
            $post = Post::where('id', $post_id)->where('user_id', $user_id)->first();
            if($post){
                $post->title = $request->title;
                $post->content = $request->content;
                $post->save();
                if($post == true){
                    $images = array();
                    if($files = $request->file('images')){
                        foreach($files as $file){
                            $image_name = uniqid() . '.' . $file->extension();
                            $file->move(public_path('uploads/post'),$image_name);
                            $images[] = $image_name;
                            $post_image = new PostImages();
                            $post_image->post_id = $post->id;
                            $post_image->image_path = $image_name;
                            $post_image->status = 1;
                            $post_image->save();
                        }
                    }
                    $response = ['status' => true, 'data' => $post, 'message' => "Success!"];
                    return response($response, 400);    
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Something went wrong."];
                    return response($response, 400);
                } 
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function delete_post(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $post_id = $request->id;
            $user_id = Auth::user()->id;
            $post = Post::where('id', $post_id)->where('user_id', $user_id)->first();
            if($post){
                if($post->user_id == $user_id){
                    $post->is_deleted = 1;
                    $post->save();
                    if($post == true){
                        $response = ['status' => true, 'data' => $post, 'message' => "Success!"];
                        return response($response, 200);  
                    }else{
                        $response = ['status' => false, 'data' => null, 'message' => "Something went wrong."];
                        return response($response, 400);
                    }
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Something went wrong."];
                    return response($response, 400);
                } 
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_post(Request $request){
        try {
            $user_id = Auth::user()->id;
            $posts = Post::orderBy('id', 'desc')->where('is_deleted',0)->with('author')->with('images')->get();
            $post_likes = PostLikes::where('user_id', $user_id)->get();
            $data_posts = array();
            $i = 0;
            foreach($posts as $post){
                $data_posts[$i] = $post;
                $data_posts[$i]['is_liked'] = false;
                foreach($post_likes as $post_like){
                    if($post_like->post_id == $post->id){
                        $data_posts[$i]['is_liked'] = true;
                    }
                }
                $i++;
            }
            $response = ['status' => true, 'data' => $data_posts, 'message' => "Success!"];
            return response($response, 200);    
             
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function like_post(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'post_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $post = Post::Where('id',$request->post_id)->first();
            if($post){
                //post does exist
                if($post->user_id == $user_id){
                    //post author cannot like its own post
                    $response = ['status' => false, 'data' => null, 'message' => "Invalid Request by same user."];
                    return response($response, 400);
                }else{
                    $is_exist = PostLikes::where('post_id',$request->post_id)->where('user_id', $user_id)->first();
                    if($is_exist){
                        //do not update
                        $response = ['status' => false, 'data' => null, 'message' => "Already Liked."];
                        return response($response, 200); 
                    }else{
                        //not post author
                        $post_likes = new PostLikes();
                        $post_likes->user_id = $user_id;
                        $post_likes->post_id = $request->post_id;
                        $post_likes->save();
                        if($post_likes == true){
                            //add to totalLikeCount
                            $post->likes_count = $post->likes_count + 1;
                            $post->save();
                            $response = ['status' => true, 'data' => $post_likes, 'message' => "Success!"];
                            return response($response, 200);    
                        }else{
                            $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                            return response($response, 400);
                        }
                    }
                }
            }else{
                //post does not exist
                $response = ['status' => false, 'data' => null, 'message' => "Invalid Request."];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function unlike_post(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'post_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $post = Post::Where('id',$request->post_id)->first();
            if($post){
                //post does exist
                $delete_post_like = PostLikes::where('post_id',$request->post_id)->where('user_id',$user_id)->delete();
                if($delete_post_like){
                    $post->likes_count = $post->likes_count - 1;
                    $post->save();
                    $response = ['status' => true, 'data' => null, 'message' => "Deleted."];
                    return response($response, 200);
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                    return response($response, 400);
                }
            }else{
                //post does not exist
                $response = ['status' => false, 'data' => null, 'message' => "Invalid Request."];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function comment_add(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'comment' => 'required',
                'post_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $post = Post::where('id', $request->post_id)->first();
            if($post){
                $comment = new Comment();
                $comment->user_id = $user_id;
                $comment->post_id = $request->post_id;
                $comment->comment = $request->comment;
                $comment->status = 1;
                $comment->save();
                if($comment == true){
                    $post->comment_count = $post->comment_count + 1;
                    $post->save();
                    $response = ['status' => true, 'data' => $comment, 'message' => "Success"];
                    return response($response, 200);    
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                    return response($response, 400);
                }
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Invalid post request."];
                return response($response, 400);
            } 
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_comment(Request $request){
        try {
            $post_id = $request->input('post_id');
            if($post_id){
                $post = Post::where("id", $post_id)->first();
                if($post){
                    $comments = Comment::where('post_id', $post_id)->orderBy('id', 'desc')->get();
                    $response = ['status' => true, 'data' => $comments, 'message' => "Success!"];
                    return response($response, 200);
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                    return response($response, 400);
                }
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Invalid request"];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function contact(Request $request){
        try {
            $validator = Validator::make($request->all(), [
                'full_name' => 'required',
                'email' => 'required',
                'message' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_id = Auth::user()->id;
            $contact = new Contact();
            $contact->user_id = $user_id;
            $contact->full_name = $request->full_name;
            $contact->email = $request->email;
            $contact->message = $request->message;
            $contact->status = 1;
            $contact->save();
            if($contact == true){
                $response = ['status' => true, 'data' => $contact, 'message' => "Success"];
                return response($response, 200);    
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_notification(Request $request){
        try {
            $user_id = Auth::user()->id;
            $notifications = Notification::where('user_id', $user_id)->get();
            $response = ['status' => true, 'data' => $notifications, 'message' => "Success!"];
            return response($response, 200);
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function subscription(Request $request){
        try {
            $user_id = Auth::user()->id;
            $validator = Validator::make($request->all(), [
                'package_id' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $package = Package::where('id', $request->package_id)->first();
            if($package){
                //package exist
                // add days to date
                $expiry_date = Date('y:m:d H:i:s', strtotime('+'.$package->valid_days.' days'));
                $user_subscription = Subscription::where('user_id', $user_id)->first();
                if($user_subscription){
                    //user is already subscribed to some package
                    $user_subscription->package_id = $package->id;
                    $user_subscription->request_earned = $package->request_allowed;
                    $user_subscription->total_amount = $package->price;
                    $user_subscription->valid_days = $package->valid_days;
                    $user_subscription->subscribed_on = date("Y-m-d H:i:s");
                    $user_subscription->expire_on = $expiry_date;
                    $user_subscription->package_detail = json_encode($package);
                    $user_subscription->status = 1;
                    $user_subscription->save();
                    if($user_subscription == true){
                        //add subscription history
                        $subscription_history = new SubscriptionHistory();
                        $subscription_history->user_id = $user_id;
                        $subscription_history->subscription_id = $user_subscription->id;
                        $subscription_history->package_id = $package->id;
                        $subscription_history->request_earned = $package->request_allowed;
                        $subscription_history->total_amount = $package->price;
                        $subscription_history->valid_days = $package->valid_days;
                        $subscription_history->subscribed_on = date("Y-m-d H:i:s");
                        $subscription_history->expire_on = $expiry_date;
                        $subscription_history->package_detail = json_encode($package);
                        $subscription_history->status = 1;
                        $subscription_history->save();
                        //add payment
                        $subscription_payment = new Payment();
                        $subscription_payment->user_id = $user_id;
                        $subscription_payment->subscription_id = $user_subscription->id;
                        $subscription_payment->subscription_history_id = $subscription_history->id;
                        $subscription_payment->payment_type = 'SUBSCRIPTION';
                        $subscription_payment->gateway = 'stripe';
                        $subscription_payment->total_amount = $package->price;
                        $subscription_payment->paid_amount = $request->paid_amount?$request->paid_amount:$package->price;
                        $subscription_payment->payment_status = $request->payment_status?$request->payment_status:'paid';
                        $subscription_payment->response_json = $request->payment_response?$request->payment_response:'';
                        $subscription_payment->status = 1;
                        $subscription_payment->save();
                        //add request quota 
                        $user_info = User::where('id', $user_id)->first();
                        $user_info->request_remaining = $user_info->request_remaining + $package->request_allowed;
                        $user_info->request_expiry = $expiry_date;
                        $user_info->save();
                        $response = ['status' => true, 'data' => $user_info, 'message' => "Success!"];
                        return response($response, 200);
                    }
                }else{
                    //new user subscription
                    $subscribe = new Subscription();
                    $subscribe->user_id = $user_id;
                    $subscribe->package_id = $package->id;
                    $subscribe->request_earned = $package->request_allowed;
                    $subscribe->total_amount = $package->price;
                    $subscribe->valid_days = $package->valid_days;
                    $subscribe->subscribed_on = date("Y-m-d H:i:s");
                    $subscribe->expire_on = $expiry_date;
                    $subscribe->package_detail = json_encode($package);
                    $subscribe->status = 1;
                    $subscribe->save();
                    if($subscribe == true){
                        //add subscription history
                        $subscription_history = new SubscriptionHistory();
                        $subscription_history->user_id = $user_id;
                        $subscription_history->subscription_id = $subscribe->id;
                        $subscription_history->package_id = $package->id;
                        $subscription_history->request_earned = $package->request_allowed;
                        $subscription_history->total_amount = $package->price;
                        $subscription_history->valid_days = $package->valid_days;
                        $subscription_history->subscribed_on = date("Y-m-d H:i:s");
                        $subscription_history->expire_on = $expiry_date;
                        $subscription_history->package_detail = json_encode($package);
                        $subscription_history->status = 1;
                        $subscription_history->save();
                        //add payment
                        $subscription_payment = new Payment();
                        $subscription_payment->user_id = $user_id;
                        $subscription_payment->subscription_id = $subscribe->id;
                        $subscription_payment->subscription_history_id = $subscription_history->id;
                        $subscription_payment->payment_type = 'SUBSCRIPTION';
                        $subscription_payment->gateway = 'stripe';
                        $subscription_payment->total_amount = $package->price;
                        $subscription_payment->paid_amount = $request->paid_amount?$request->paid_amount:$package->price;
                        $subscription_payment->payment_status = $request->payment_status?$request->payment_status:'paid';
                        $subscription_payment->response_json = $request->payment_response?$request->payment_response:'';
                        $subscription_payment->status = 1;
                        $subscription_payment->save();
                        //add request quota 
                        $user_info = User::where('id', $user_id)->first();
                        $user_info->request_remaining = $user_info->request_remaining + $package->request_allowed;
                        $user_info->request_expiry = $expiry_date;
                        $user_info->save();
                        $response = ['status' => true, 'data' => $user_info, 'message' => "Success!"];
                        return response($response, 200);
                    }else{
                        $response = ['status' => false, 'data' => null, 'message' => "Something went wrong please try again later."];
                        return response($response, 400);
                    }
                }
            }else{
                //package does not exist
                $response = ['status' => false, 'data' => null, 'message' => "Invalid Request."];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function get_all_settings(){
        try {
            $settings = Settings::orderBy('id', 'desc')->get();
            $response = ['status' => true, 'data' => $settings, 'message' => "Success!"];
            return response($response, 200);    
             
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }

    public function boost(Request $request){
        try {
            $user_id = Auth::user()->id;
            $validator = Validator::make($request->all(), [
                'valid_days' => 'required',
                'valid_from' => 'required',
                'valid_to' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()]);
            }
            $user_info = User::where('id', $user_id)->first();
            if($user_info){
                //package exist
                // add days to date
                $boostProfile = new BoostProfile();
                $boostProfile->user_id = $user_id;
                $boostProfile->valid_days = $request->valid_days;
                $boostProfile->valid_from = date('Y-m-d', strtotime($request->valid_from));
                $boostProfile->valid_to = date('Y-m-d', strtotime($request->valid_to));
                $boostProfile->status = 1;
                $boostProfile->save();
                if($boostProfile){
                    $response = ['status' => true, 'data' => $boostProfile, 'message' => "Success!"];
                    return response($response, 200);
                }else{
                    $response = ['status' => false, 'data' => null, 'message' => "Something Went Wrong"];
                    return response($response, 400);    
                }
            }else{
                $response = ['status' => false, 'data' => null, 'message' => "Invalid Request."];
                return response($response, 400);
            }
        } catch (Exception $e) {
            $response = ['status' => false, 'data' => null, 'message' => $e->getMessage()];
            return response($response, 400);
        }
    }
}
